import axios from 'axios';

// Configure axios for Sanctum
const apiClient = axios.create({
  baseURL: import.meta.env.VITE_API_BASE_URL || window.location.origin,
  withCredentials: true,
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
    'X-Requested-With': 'XMLHttpRequest'
  }
});

const LOCAL_STORAGE_KEY = 'auth';

// Auth store
const authStore = {
  namespaced: true,
  state: {
    user: JSON.parse(localStorage.getItem(LOCAL_STORAGE_KEY))?.user || null,
    authenticated: !!JSON.parse(localStorage.getItem(LOCAL_STORAGE_KEY))?.auth || false,
    loading: false,
    error: null
  },
  mutations: {
    SET_USER(state, user) {
      state.user = user;
      state.authenticated = !!user;

      if (user) {
        localStorage.setItem(
          LOCAL_STORAGE_KEY,
          JSON.stringify({ auth: true, user })
        );
      } else {
        localStorage.removeItem(LOCAL_STORAGE_KEY);
      }
    },
    CLEAR_AUTH(state) {
      state.user = null;
      state.authenticated = false;
      localStorage.removeItem(LOCAL_STORAGE_KEY);
    },
    SET_LOADING(state, isLoading) {
      state.loading = isLoading;
    },
    SET_ERROR(state, error) {
      state.error = error;
    }
  },
  actions: {
    async initialize({ commit, dispatch }) {
      commit('SET_LOADING', true);
      try {
        await dispatch('checkAuth');
      } finally {
        commit('SET_LOADING', false);
      }
    },

    async login({ commit }, credentials) {
      commit('SET_LOADING', true);
      commit('SET_ERROR', null);

      try {
        await apiClient.get('/sanctum/csrf-cookie');
        const loginResponse = await apiClient.post('/login', credentials);
        
        // Fetch user data after successful login
        const { data } = await apiClient.get('/api/user');
        commit('SET_USER', data);
        
        return data;
      } catch (error) {
        const message = error.response?.data?.message || 'Login failed';
        commit('SET_ERROR', message);
        throw error;
      } finally {
        commit('SET_LOADING', false);
      }
    },

    async logout({ commit }) {
      try {
        await apiClient.post('/logout');
      } catch (_) {}
      commit('CLEAR_AUTH');
      window.location.href = '/login';
    },

    async checkAuth({ commit }) {
      try {
        const { data } = await apiClient.get('/api/user');
        commit('SET_USER', data);
        return true;
      } catch (_) {
        commit('CLEAR_AUTH');
        return false;
      }
    },

    async fetchUser({ commit }) {
      try {
        const { data } = await apiClient.get('/api/user');
        commit('SET_USER', data);
        return data;
      } catch (error) {
        commit('CLEAR_AUTH');
        throw error;
      }
    }
  },
  getters: {
    isAuthenticated: state => state.authenticated,
    currentUser: state => state.user,
    isLoading: state => state.loading,
    error: state => state.error
  }
};

// Response interceptor - now has access to store through a different approach
apiClient.interceptors.response.use(
  response => response,
  error => {
    // Check if the error is 401 (Unauthorized) or 419 (CSRF token mismatch)
    // AND ensure the request wasn't checking simple auth status (which is expected to fail for guests)
    const isAuthCheck = error.config?.url?.includes('/api/user');
    
    if ((error.response?.status === 401 || error.response?.status === 419) && !isAuthCheck) {
      // Clear localStorage directly since we can't access store here easily
      localStorage.removeItem(LOCAL_STORAGE_KEY);
      if (!window.location.pathname.includes('login')) {
        window.location.href = '/login';
      }
    }
    return Promise.reject(error);
  }
);

export default authStore;