<?php

namespace App\Http\Controllers;

use App\Models\Portfolio;
use Illuminate\Http\Request;

class PortfolioController extends Controller
{
    public function index()
    {
        try {
            $projects = Portfolio::latest()->get();
            return response()->json([
                'success' => true,
                'data' => $projects
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch projects',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'short_description' => 'required|string',
                'long_description' => 'required|string',
                'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
                'link' => 'nullable|url',
                'client' => 'nullable|string|max:255',
                'technologies' => 'nullable|string',
                'category_id' => 'required|exists:portfolio_categories,id',
                'is_featured' => 'nullable|boolean',
                'is_active' => 'nullable|boolean'
            ]);

            // Store image
            $imagePath = $request->file('image')->store('portfolio', 'public');

            $project = Portfolio::create([
                'title' => $validated['title'],
                'short_description' => $validated['short_description'],
                'long_description' => $validated['long_description'],
                'image' => $imagePath,
                'link' => $validated['link'] ?? null,
                'client' => $validated['client'] ?? null,
                'technologies' => $validated['technologies'] ?? null,
                'category_id' => $validated['category_id'],
                'is_featured' => $validated['is_featured'] ?? false,
                'is_active' => $validated['is_active'] ?? true,
            ]);

            return response()->json([
                'success' => true,
                'data' => $project,
                'message' => 'Project created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create project',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, Portfolio $portfolio)
    {
        $validated = $request->validate([
            'title' => 'sometimes|string|max:255',
            'short_description' => 'sometimes|string',
            'long_description' => 'sometimes|string',
            'image' => 'sometimes|image|mimes:jpeg,png,jpg,gif|max:2048',
            'link' => 'nullable|url',
            'client' => 'nullable|string|max:255',
            'technologies' => 'nullable|string',
            'category_id' => 'required|exists:portfolio_categories,id',
            'is_featured' => 'sometimes|boolean',
            'is_active' => 'sometimes|boolean'
        ]);

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('portfolio', 'public');
            $validated['image'] = $imagePath;
        }

        $portfolio->update($validated);

        return $portfolio; // apiResource expects the updated resource
    }

    public function destroy(Portfolio $portfolio)
    {
        $portfolio->delete();
        return response()->noContent(); // Standard for DELETE requests
    }
}
