<?php

namespace App\Http\Controllers;

use App\Models\PortfolioCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PortfolioCategoryController extends Controller
{
    public function index()
    {
        return PortfolioCategory::orderBy('order')->get();
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:portfolio_categories,slug',
            'icon' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'order' => 'nullable|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048'
        ]);

        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('portfolio_categories', 'public');
        }

        PortfolioCategory::create($data);

        return response()->json(['message' => 'Category created successfully']);
    }

    public function update(Request $request, PortfolioCategory $portfolioCategory)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:portfolio_categories,slug,' . $portfolioCategory->id,
            'icon' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'order' => 'nullable|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048'
        ]);

        if ($request->hasFile('image')) {
            // Optional: delete old image
            if ($portfolioCategory->image) {
                Storage::disk('public')->delete($portfolioCategory->image);
            }
            $data['image'] = $request->file('image')->store('portfolio_categories', 'public');
        }

        $portfolioCategory->update($data);

        return response()->json(['message' => 'Category updated successfully']);
    }

    public function destroy(PortfolioCategory $portfolioCategory)
    {
        // Optional: delete image file
        if ($portfolioCategory->image) {
            Storage::disk('public')->delete($portfolioCategory->image);
        }

        $portfolioCategory->delete();

        return response()->json(['message' => 'Category deleted successfully']);
    }
}
