<?php

namespace App\Http\Controllers;

use App\Models\Map;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class MapController extends Controller
{
    /**
     * Display a listing of the maps for admin dashboard.
     */
    public function index(): JsonResponse
    {
        try {
            $maps = Map::orderBy('sort_order')->get();

            return response()->json([
                'success' => true,
                'data' => $maps
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@index error: ' . $e->getMessage());
            return $this->errorResponse('Failed to fetch maps');
        }
    }

    /**
     * Get active maps for frontend contact page
     */
    public function publicLocations(): JsonResponse
    {
        try {
            $maps = Map::active()->ordered()->get();

            return response()->json([
                'success' => true,
                'data' => $maps
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@publicLocations error: ' . $e->getMessage());
            return $this->errorResponse('Failed to fetch map locations');
        }
    }

    /**
     * Store a newly created map in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'required|string',
            'description' => 'required|string',
            'map_url' => [
                'required',
                'url',
                'regex:/^https:\/\/www\.google\.com\/maps\/embed\?/',
            ],
            'is_active' => 'boolean',
            'sort_order' => 'integer',
        ]);

        Map::create($validated);

        return response()->json(['message' => 'Map saved successfully'], 201);
    }

    /**
     * Display the specified map.
     */
    public function show(Map $map): JsonResponse
    {
        try {
            return response()->json([
                'success' => true,
                'data' => $map
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@show error: ' . $e->getMessage());
            return $this->errorResponse('Map not found', 404);
        }
    }

    /**
     * Update the specified map in storage.
     */
    public function update(Request $request, Map $map): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'address' => 'sometimes|string',
            'description' => 'sometimes|string',
            'map_url' => 'sometimes|url',
            'is_active' => 'sometimes|boolean',
            'sort_order' => 'sometimes|integer'
        ]);

        try {
            $map->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Map updated successfully',
                'data' => $map
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@update error: ' . $e->getMessage());
            return $this->errorResponse('Failed to update map', 500);
        }
    }

    /**
     * Remove the specified map from storage.
     */
    public function destroy(Map $map): JsonResponse
    {
        try {
            $map->delete();

            return response()->json([
                'success' => true,
                'message' => 'Map deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@destroy error: ' . $e->getMessage());
            return $this->errorResponse('Failed to delete map', 500);
        }
    }

    /**
     * Toggle the active status of a map.
     */
    public function toggleStatus(Map $map): JsonResponse
    {
        try {
            $map->update(['is_active' => !$map->is_active]);

            return response()->json([
                'success' => true,
                'message' => 'Map status updated successfully',
                'data' => $map
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@toggleStatus error: ' . $e->getMessage());
            return $this->errorResponse('Failed to update map status', 500);
        }
    }

    /**
     * Get active locations for admin UI
     */
    public function activeLocations(): JsonResponse
    {
        try {
            $maps = Map::active()->ordered()->get();

            return response()->json([
                'success' => true,
                'data' => $maps
            ]);
        } catch (\Exception $e) {
            Log::error('MapController@activeLocations error: ' . $e->getMessage());
            return $this->errorResponse('Failed to fetch active locations');
        }
    }

    /**
     * Helper method for error responses
     */
    private function errorResponse(string $message, int $status = 500): JsonResponse
    {
        return response()->json([
            'success' => false,
            'message' => $message,
            'errors' => []
        ], $status);
    }
}
